/***************************************************************
 *                   Materials Object Library                  *
 *  Class orthotropic_tensor: decl. for orthotropic materials  *
 *                    simula+@metz.ensam.fr                    *
 *                   GNU/linux version 2.6.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2007,2008 COLLARD Christophe
 * copyright © 2007,2008 Centre National de la Recherche Scientifique
 * copyright © 2007,2008 Arts et Métiers ParisTech
 * copyright © 2007,2008 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 ***************************************************************/

/*! \namespace materiol
    \brief Materials Object Libraries
*/

/*! \class materiol::orthotropic_tensor
    \brief \f$ 4^{th} \f$ order orthotropic tensor computation for orthotropic materials

    \htmlonly 
    <FONT color="#838383">

    orthotropic tensor belongs to Materials Object Libraries (MateriOL++) </br>
    MateriOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    We define an orthotropic tensor \f$ C \f$ through nine constants : \f$ c_{11}, c_{22}, c_{33}, c_{44}, c_{55}, c_{66}, c_{12}, c_{13}, c_{23} \f$. Its non zero components read \n
    \f$ \displaystyle C_{1111} = c_{11} \f$ \n
    \f$ \displaystyle C_{2222} = c_{22} \f$ \n
    \f$ \displaystyle C_{3333} = c_{33} \f$ \n
    \f$ \displaystyle C_{1212} = C_{1221} = C_{2112} = C_{2121} = c_{44} \f$ \n
    \f$ \displaystyle C_{1313} = C_{1331} = C_{3113} = C_{3131} = c_{55} \f$ \n
    \f$ \displaystyle C_{2323} = C_{2332} = C_{3223} = C_{3232} = c_{66} \f$ \n
    \f$ \displaystyle C_{1122} = C_{2211} = c_{12} \f$ \n
    \f$ \displaystyle C_{1133} = C_{3311} = c_{13} \f$ \n
    \f$ \displaystyle C_{2233} = C_{3322} = c_{23} \f$ \n

    The associated reduced symmetric matrix reads: \n
     \f$  \begin{pmatrix} c_{11} & c_{12} & c_{13} & 0      & 0      & 0 \\ 
                                 & c_{22} & c_{23} & 0      & 0      & 0 \\
			         &        & c_{33} & 0      & 0      & 0 \\
			         & sym    &        & c_{44} & 0      & 0 \\
			         &        &        &        & c_{55} & 0 \\
			         &        &        &        &        & c_{66} \\
        \end{pmatrix} 
	\f$ \n
	and we get the following symmetries : \f$C_{ijkl} = C_{jikl} = C_{ijlk} = C_{klij} \f$.

    \author copyright \htmlonly &#169; \endhtmlonly 2007, 2008 Christophe COLLARD \n
            copyright \htmlonly &#169; 2007, 2008 Centre National de la Recherche Scientifique \endhtmlonly \n
	    copyright \htmlonly &#169; 2007, 2008 Arts et M&#233;tiers ParisTech \endhtmlonly \n
            copyright \htmlonly &#169; 2007, 2008 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
    \version 2.6.0
    \date 2007-2011
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type orthotropic tensor
#endif

#ifndef _orthotropic_tensors_h
#define _orthotropic_tensors_h


#if !defined (__IOSTREAM_H)
#include <iostream>
#endif

#if !defined(__FSTREAM_H)
#include <fstream>
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__MATHS_H)
#include <math.h>
#endif

#if !defined(__STRING_H)
#include <string.h>
#endif

#if !defined(__VECTORS_H)
#include "MOL++/vectors.h"
#endif

#if !defined(__TENSORS4_H)
#include "MOL++/tensors4.h"
#endif

using namespace std;
using namespace mol;

namespace materiol
{


//=============================================================
template <class T> class orthotropic_tensor : public tensor4<T>
//=============================================================
{
  using tensor4<T>::size;
  using tensor4<T>::tsr;

  protected :
    T C11, C22, C33, C44, C55, C66, C12, C13, C23; // material constants
    string tensor_name;

    void build_tensor ();  // computes the 4th order isotropic tensor

  public :
    orthotropic_tensor (string = "");  // default constructor
    orthotropic_tensor (T, T, T, T, T, T, T, T, T, string = "");
    orthotropic_tensor (const vector<T>&, string = "");
    orthotropic_tensor (const tensor4<T>&, string = "");  // cast convertion
    ~orthotropic_tensor () {}  // destructor

    string name(string = "");  // returns / computes tensor's name
    vector<T> elastic_constants ();
    orthotropic_tensor<T> inv ();

    template <class Tf> friend ostream& operator << (ostream&, orthotropic_tensor<Tf>&);
};


//=====Private methods for orthotropic tensor===========================================


//------------------------------------------------------------
template <class T> void orthotropic_tensor<T>::build_tensor ()
//------------------------------------------------------------
{
  if (!size)  // used for de derivated classes
    { tsr = new tensor3<T> [size=3];
      for (int i=0; i<size; i++)
	tsr[i] =& tensor3<T> (size,size,size);
    }

  (*this)(1,1,1,1) = C11;
  (*this)(2,2,2,2) = C22;
  (*this)(3,3,3,3) = C33;
  (*this)(1,2,1,2) = (*this)(1,2,2,1) = (*this)(2,1,1,2) = (*this)(2,1,2,1) = C44;
  (*this)(1,3,1,3) = (*this)(1,3,3,1) = (*this)(3,1,1,3) = (*this)(3,1,3,1) = C55;
  (*this)(2,3,2,3) = (*this)(2,3,3,2) = (*this)(3,2,2,3) = (*this)(3,2,3,2) = C66;
  (*this)(1,1,2,2) = (*this)(2,2,1,1) = C12;
  (*this)(1,1,3,3) = (*this)(3,3,1,1) = C13;
  (*this)(2,2,3,3) = (*this)(3,3,2,2) = C23;
}


//=====Public methods for cubic tensor==================================================


/*!
  \brief Constructor for \f$ \displaystyle 4^{th} \f$ order orthotropic tensors.

  \param name tensor name
*/

//------------------------------------------------------------------------
template <class T> orthotropic_tensor<T>::orthotropic_tensor (string name)
//------------------------------------------------------------------------
{
  C11 = C22 = C33 = C44 = C55 = C66 = C12 = C13 = C23 = 0;
  tensor_name = name;
}


/*!
  \brief Constructor for \f$ \displaystyle 4^{th} \f$ order orthotropic tensors.

  Computes \f$ C_{ijkl} \f$ (see detailed description).


  \param c11 \f$ c_{11} \f$ value
  \param c22 \f$ c_{22} \f$ value
  \param c33 \f$ c_{33} \f$ value
  \param c44 \f$ c_{44} \f$ value
  \param c55 \f$ c_{55} \f$ value
  \param c66 \f$ c_{66} \f$ value
  \param c12 \f$ c_{12} \f$ value
  \param c13 \f$ c_{13} \f$ value
  \param c23 \f$ c_{23} \f$ value
  \param name tensor name
*/

//-------------------------------------------------------------------------------------------------------------------------------------------------------
template <class T> orthotropic_tensor<T>::orthotropic_tensor (T c11, T c22, T c33, T c44, T c55, T c66, T c12, T c13, T c23, string name) : tensor4<T>(3)
//-------------------------------------------------------------------------------------------------------------------------------------------------------
{
  C11 = c11;
  C22 = c22;
  C33 = c33;
  C44 = c44;
  C55 = c55;
  C66 = c66;
  C12 = c12;
  C13 = c13;
  C23 = c23;
  build_tensor();
  tensor_name = name;
}


/*!
  \brief Constructor for \f$ \displaystyle 4^{th} \f$ order orthotropic tensors.

  Computes \f$ C_{ijkl} \f$ (see detailed description).


  \param v \f$ (c_{11} , c_{22}, c_{33}, c_{44} , c_{55}, c_{66}, c_{12}, c_{13}, c_{23}) \f$ values
  \param name tensor name
*/

//------------------------------------------------------------------------------------------------------------
template <class T> orthotropic_tensor<T>::orthotropic_tensor (const vector<T>& v, string name) : tensor4<T>(3)
//------------------------------------------------------------------------------------------------------------
{
  assert (v.dim() == 9);

  C11 = v[1];
  C22 = v[2];
  C33 = v[3];
  C44 = v[4];
  C55 = v[5];
  C66 = v[6];
  C12 = v[7];
  C13 = v[8];
  C23 = v[9];
  build_tensor();
  tensor_name = name;
}


/*!
  \brief Constructor for cast conversion.

  Converts a \f$ \displaystyle 4^{th} \f$ order tensor into an orthotropic tensor.

  \param t \f$ \displaystyle 4^{th} \f$ order tensor
  \param name tensor name
*/

//---------------------------------------------------------------------------------------------
template <class T> orthotropic_tensor<T>::orthotropic_tensor (const tensor4<T>& t, string name)
//---------------------------------------------------------------------------------------------
{
  assert(t.dim1()==3 && t.dim2()==3 && t.dim3()==3 && t.dim4()==3);

  C11 = t(1,1,1,1);
  C22 = t(2,2,2,2);
  C33 = t(3,3,3,3);
  C44 = t(1,2,1,2);
  C55 = t(1,3,1,3);
  C66 = t(2,3,2,3);
  C12 = t(1,1,2,2);
  C13 = t(1,1,3,3);
  C23 = t(2,2,3,3);
  tensor_name = name;
  build_tensor();
  assert ((tensor4<T>)(*this) == t);
}


/*!
  \brief Changes and/or returns tensor name
  \param name tensor name
  \return tensor name
*/

//-----------------------------------------------------------------
template <class T> string orthotropic_tensor<T>::name (string name)
//-----------------------------------------------------------------
{
  if (name != "") tensor_name = name;

  return tensor_name;
}


/*!
  \brief Returns elastic constants.
  \return \f$ (c_{11}, c_{22}, c_{33} ,c_{44}, c_{55}, c_{66}, c_{12}, c_{13}, c_{23}) \f$
*/

//----------------------------------------------------------------------
template <class T> vector<T> orthotropic_tensor<T>::elastic_constants ()
//----------------------------------------------------------------------
{
  vector<T> constants (9,false);
  constants[1] = C11;
  constants[2] = C22;
  constants[3] = C33;
  constants[4] = C44;
  constants[5] = C55;
  constants[6] = C66;
  constants[7] = C12;
  constants[8] = C13;
  constants[9] = C23;

  return constants;
}


/*!
  \brief Computes the inverse of a \f$ \displaystyle 4^{th} \f$ order orthotropic tensor.

  If we note \f$ S = C^{-1} \f$, we can compute \f$S\f$ using a bloc decomposition of the matrix associated with \f$C\f$ and compute more easily its inverse to obtain the matrix associated with \f$ S \f$. Thus the non zero elements of \f$ S \f$ read : \n
  \f$ \displaystyle S_{1111} = s_{11} = \frac{1}{\Delta} (c_{22}c_{33} - c_{23}^2) \f$ \n
  \f$ \displaystyle S_{2222} = s_{22} = \frac{1}{\Delta} (c_{11}c_{33} - c_{13}^2) \f$ \n
  \f$ \displaystyle S_{3333} = s_{33} = \frac{1}{\Delta} (c_{11}c_{22} - c_{12}^2) \f$ \n
  \f$ \displaystyle S_{1212} = S_{1221} = S_{2112} = S_{2121} = s_{44} = \frac{1}{c_{44}} \f$ \n
  \f$ \displaystyle S_{1313} = S_{1331} = S_{3113} = S_{3131} = s_{55} = \frac{1}{c_{55}} \f$ \n
  \f$ \displaystyle S_{2323} = S_{2332} = S_{3223} = S_{3232} = s_{66} = \frac{1}{c_{66}} \f$ \n
  \f$ \displaystyle S_{1122} = S_{2211} = s_{12} = s_{21} = - \frac{1}{\Delta} (c_{12}c_{33} - c_{13}c_{23}) \f$ \n
  \f$ \displaystyle S_{1133} = S_{3311} = s_{13} = s_{31} = \frac{1}{\Delta} (c_{12}c_{23} - c_{22}c_{13}) \f$ \n
  \f$ \displaystyle S_{2233} = S_{3322} = s_{23} = s_{32} = - \frac{1}{\Delta} (c_{11}c_{23} - c_{12} c_{13}) \f$ \n
  where
  \f$ \Delta = c_{11} (c_{22}c_{33} - c_{23}^2) - c_{12} (c_{12}c_{33} - c_{13}c_{23}) + c_{13} (c_{12}c_{23} - c_{22}c_{13}) \f$.
  \return inverse tensor
*/

//--------------------------------------------------------------------
template <class T> orthotropic_tensor<T> orthotropic_tensor<T>::inv ()
//--------------------------------------------------------------------
{
  T D1 = C22*C33 - pow(C23,2);
  T D2 = C13*C23 - C12*C33;
  T D3 = C12*C23 - C22*C13;
  T delta = C11*D1 + C12*D2 + C13*D3;

  assert (abs(delta) > epsilon);
  assert (abs(C44)>epsilon && abs(C55)>epsilon && abs(C66)>epsilon);

  T delta_inv = 1/delta;

  return orthotropic_tensor<T> (D1*delta_inv, (C11*C33-pow(C13,2))*delta_inv, (C11*C22-pow(C12,2))*delta_inv, .25/C44, .25/C55, .25/C66, D2*delta_inv, D3*delta_inv, (C12*C13-C11*C23)*delta_inv); // codage (symtensor -> matrix) used here is not the same that the one of the documentation => please do not modify 0.25 / C44, 0.25 / C55 and 0.25 / C66 !!
}


//--------------------------------------------------------------------------------
template <class Tf> ostream& operator << (ostream& s, orthotropic_tensor<Tf>& tsr)
//--------------------------------------------------------------------------------
{
  assert (tsr.C11 * tsr.C22 * tsr.C33 * tsr.C44 * tsr.C55 * tsr.C66 * tsr.C12 * tsr.C13 * tsr.C23);

  s << tsr.tensor_name << endl;
  s << "Material constants : (" << tsr.C11 << ", " << tsr.C22 << ", " << tsr.C33 << ", " << tsr.C44 << ", " << tsr.C55 << ", " << tsr.C66 << ", " << tsr.C12 << ", " << tsr.C13 << ", " << tsr.C23 << ")" << endl;
  s << (tensor4<Tf>) tsr;
  s << endl;

  return s;
}


}


#endif
